; =====================================================================
; Anzeigemodul mit AT90S2313 fr Temperaturmessystem
; =====================================================================
;
; Version 1.06, letzte Bearbeitung am 24.10.2006
;
; ***********************************************
; * Modifizierte Version der Software 1.06      *
; * fr die Benutzung von LCD-Displays mit dem  *
; * KS0073 Controller (IE=HIGH).		*
; * (Originalprojekt verwendet ein Display      *
; * mit KS0066(u) ) 				*
; * nderungen und zustzliche Kommentierungen  *
; * sind mit ;;;<--> gekennzeichnet.            *
; *						* 
; * nderungen von Max Zimmermann (c) 2008      *
; * http://www.schlaflos-in-mainz.de            *
; * max@schlaflos-in-mainz.de                   *
; * GPG Key ist auf der Website zu finden       *
; ***********************************************
;
; Dieses Anzeigemodul verarbeitet die vom Sensormodul empfangenen
; Datenpakete im folgenden ASCII Format:
;
; * Sensorbelegung:
;
; 33000200#     (Beispiel: 3 Sensoren an Bus1, 3 Sensoren an Bus2 und
;		2 Sensoren an Bus6)
;	  - Enter-Zeichen CR (13)
; --------  Anzahl der Sensoren an jedem 1-Wire-Bus, von Bus1 bis Bus8
;
; Diese Meldung wird nur einmal in der Startphase des Sensormoduls
; gesendet (ca. 11 Sekunden nach dem Einschalten).
;
; * Temperaturwerte:
;
; 2: 23.5#	(Beispiel: Temperatur 2 = 23.5C)
;        - Enter-Zeichen CR (13)
;   -----  Temperaturwert im Bereich -55.0 bis 125.0
;  -       Trennzeichen (Doppelpunkt)
; -        Sensornummer im Bereich 1-8 oder a-h (entspricht 9-16)
;
; Anstatt des Dezimalpunktes wird auch ein Komma akzeptiert. Das Minus-
; zeichen kann an 3. oder 4. Stelle stehen. Werden keine Temperatur-
; werte mehr empfangen, so wird nach einem Timeout von 60 Sekunden der
; Wert des betreffenden Sensors vom LCD gelscht
;
; * Alarmmeldungen:
;
; C:1#		(Beispiel: Alarm 3 ein)
;    - Enter-Zeichen CR (13)
;   -  Alarmwert 0=Alarm aus, 1=Alarm ein
;  -   Trennzeichen (Doppelpunkt)
; -    Alarmnummer im Bereich A-D
;
; Allgemein gilt: Werden keine Datenpakete mehr empfangen, so wird nach
; einem Timeout von 20 Sekunden die Meldung "kein Datenempfang" auf dem
; LCD angezeigt.
;
; =====================================================================
;
; ----------------------
; Belegung der I/O-Ports
; ----------------------
;
; PortB0: Ausgang LCD-Anschluss D0 \
; PortB1: Ausgang LCD-Anschluss D1  \
; PortB2: Ausgang LCD-Anschluss D2   \
; PortB3: Ausgang LCD-Anschluss D3    \ Standard-LCD Anschluss mit
; PortB4: Ausgang LCD-Anschluss D4    / HD44780 oder KS0066 Controller
; PortB5: Ausgang LCD-Anschluss D5   /
; PortB6: Ausgang LCD-Anschluss D6  /
; PortB7: Ausgang LCD-Anschluss D7 /
;
; PortD0: Eingang mit Pull-up, RS232-RXD, Empfangsdaten, Baudrate 9600
; PortD1: Eingang mit Pull-up, Taste
; PortD2: Ausgang LCD-Anschluss RS
; PortD3: Ausgang LCD-Anschluss E
; PortD4: Ausgang LED (low=ein)
; PortD5: Ausgang Signalgeber (high=ein)
; PortD6: Ausgang (nicht genutzt)
;
; ------------------
; Belegung der Timer
; ------------------
;
; Timer0: nicht genutzt
; Timer1: generiert Interrupts alle 20 ms zur Steuerung der Timeouts
;         und zur Tastenentprellung
;
.nolist
.include "2313def.inc"
.list
;
.equ	clock=	4000000	;Taktfrequenz= 4,0 MHz
.equ	timval=	10000	;Timer1-Wert fr 50 Hz / 20 ms bei Vorteiler 8
.equ	baud=	9600	;Baudrate= 9600
.equ	timtmp=	60	;Timeout-Wert fr Temperaturwerte in Sekunden
			;(eingestellt auf 60 Sekunden)
.equ	timmsg=	20	;Timeout-Wert fr Alarmmeldungen in Sekunden
			;(eingestellt auf 20 Sekunden)
.equ	timsem=	10	;Timeout-Wert fr Sensorbelegungs-Anzeige in
			;Sekunden (eingestellt auf 10 Sekunden)
.equ	timbuz=	150	;Timeout-Wert fr Buzzer in 1/50 Sekunden,
			;sollte ein Vielfaches von 50 betragen,
			;(eingestellt auf 3,0 Sekunden)
.equ	timdim=	30	;Timeout fr Anzeigemodus - Rckschaltung auf
			;Temperaturanzeige nach Alarmrckstellung
			;(eingestellt auf 30 Sekunden)
.equ	cr=	13	;Zeichencode Cariage Return (Enter)
.equ	lf=	10	;Zeichencode fr Line Feed
.equ	led=	4	;LED-Bit von PortD
.equ	buzzer=	5	;Piezo-Summer-Bit von PortD
;
.def	insreg=	r1	;Zwischenspeicher fr SREG whrend Interrupt
.def	rxpoi1=	r2	;UART Empfangspuffer-Pointer1 (Schreibposition)
.def	rxpoi2=	r3	;UART Empfangspuffer-Pointer2 (Leseposition)
.def	delayc=	r4	;Zeitschleifenzhler
.def	rspoin=	r5	;Zeiger auf RS-232 Textpuffer
.def	precnt=	r6	;Interrupt-Vorteiler (20 ms -> 1 sek)
.def	buzcn1=	r7	;Timeout-Zhler fr Buzzer, wird vom Hauptpro-
			;gramm auf einen Wert gesetzt und vom Interrupt
			;im Takt von 20ms auf 0 gezhlt
.def	buzcn2=	r8	;wie r7, fr Temperatur-Alarm
.def	secnt1=	r9	;Sekundenzhler, wird fr LCD-Blinken bentigt
.def	secnt2=	r10	;Zwischenspeicher fr Sekundenzhler
.def	msgtio=	r11	;Timeout-Zhler fr Meldungen; wird bei Empfang
			;eines Datensatzes auf 20 sek gesetzt und vom
			;Timer-Interrupt auf 0 gezhlt; Hauptprogramm
			;lscht bei 0 den Temperaturwert auf dem LCD
			;und setzt den Wert auf 0xff (Stopp)
.def	dimtio=	r12	;Timeout-Zhler fr die Rckkehr zur Tempera-
			;turanzeige nach einer Alarmauslsung
.def	lastal=	r13	;Alarmnummer des zuletzt ausgelsten Alarms,
			;wird fr Alarmton bentigt
.def	dimode=	r14	;Anzeige-Modus
			;0= Temperaturanzeige
			;1= Alarmanzeige
			;2= Anzeige der Sensorbelegung
.def	flags=	r24	;Verschiedene Flags
			;0= nderungsflag 1, wird bei nderungen von
			;   Temperaturwerten gesetzt und nach der
			;   Aktualisierung des LCD wieder gelscht
			;1= Tastenflag 1 (gedrckt), wird von Interrupt
			;   gesetzt und gelscht
			;2= Tastenflag 2 (entprellt), wird von Inter-
			;   rupt gesetzt und gelscht; dient als Flag
			;   fr das Hauptprogramm
			;3= Tastenflag 3 (quittiert), wird vom Haupt-
			;   programm gesetzt und von Interrupt gelscht
			;4= Fehler-Flag; wird gesetzt, wenn der Mel-
			;   dungs-Timeout abgelaufen ist, anstatt der
			;   Temperaturanzeige erscheint ein Fehlertext
			;5= nderungsflag 2, wird bei nderung von
			;   Alarmen gesetzt und nach der Aktualisierung
			;   der Alarmanzeige wieder gelscht
.def	itemp=	r25	;Zwischenspeicher whrend Interruptbearbeitung
;
; --------------------------------------------------------------------
; SRAM-Belegung
; Adressen nicht verschieben, "rxbuff" muss unbedingt auf 0x80 liegen!
; --------------------------------------------------------------------
;
.dseg
.org	0x60
;
rstext:	.byte	8	;Text-Empfangspuffer; wird vom Hauptprogramm
			;aus UART-Puffer kopiert und verarbeitet
temmap:	.byte	8	;Anzeige-Tabelle, wird aus EEPROM gelesen und
			;enthlt die LCD-Positionen von 8 mglichen
			;Temperatur-Sensoren (8 aus 16)
tempto:	.byte	8	;Timeout-Zhler fr Temperatur 1-8; wird bei
			;Empfang eines Datensatzes auf 60 sek gesetzt
			;und von Timer-Interrupt auf 0 gezhlt; Haupt-
			;programm lscht bei 0 den Temperaturwert auf
			;LCD und setzt den Wert auf 0xff (Stopp)
alminv:	.byte	4	;Alarm-Invertierungs-Tabelle, wird aus EEPROM
			;gelesen und bestimmt, ob die empfangene Alarm-
			;meldung invertiert wird oder nicht:
			;0= normal
			;1= invertiert
almbuf:	.byte	4	;Speicher fr 4 Alarme, jeder Alarm kann
			;folgende Zustnde annehmen:
			;0= kein Alarm
			;1= Alarm aktiv
rxbuff:	.byte	32	;UART Empfangspuffer (muss auf 0x80 liegen)
tembuf:	.byte	8*5	;Speicher fr die empfangenen Temperaturwerte
temlim:	.byte	8	;Flags fr Temperaturlimit-berschreitungen
			;Bit 0= Temperatur-Unterschreitung
			;Bit 1= Temperatur-berschreitung
;
; --------------
; Programmbeginn
; --------------
;
;	Interrupt-Vektoren
;
.cseg
.org	0x0000
;
reset:	rjmp	start			;Programmstart
exint0:	reti				;nicht genutzt
exint1:	reti				;nicht genutzt
t1icap:	reti				;nicht genutzt
t1comp:	rjmp	timer1			;Timer1, alle 20 ms
t1ovfl:	reti				;nicht genutzt
t0ovfl:	reti				;nicht genutzt
uarrxc:	rjmp	rxcint			;UART Empfang komplett
uartxe:	reti				;nicht genutzt
uartxc:	reti				;nicht genutzt
ancomp:	reti				;nicht genutzt
;
; Initialisierung von I/O, RAM und Registern
;
start:	cli				;Interrupts sperren
	wdr				;Watchdog zurcksetzen
	ldi	r16,(1<<wde)+(1<<wdp2)+(1<<wdp1)+(1<<wdp0)												
	out	wdtcr,r16		;Watchdog aktivieren (1,9s)
;
	ldi	zl,0x60			;Zeiger auf RAM-Anfang
	ldi	zh,0
	clr	r16
sta010:	st	z+,r16			;Speicherzelle lschen			
	cpi	zl,ramend+1		;Ende erreicht?
	brne	sta010			;nein -> Schleife
	ldi	r16,ramend							
	out	spl,r16			;Stackpointer setzen
;
	clr	r16			;PortB Ausgnge auf 0 setzen
	out	portb,r16
	ldi	r16,0b00000011		;PortD Ausgnge auf 0 setzen
	out	portd,r16		;und Eingang mit Pullup setzen
	ser	r16			;PortB als Ausgnge setzen		
	out	ddrb,r16
	ldi	r16,0b01111100		;PortD als Ausgnge setzen,
	out	ddrd,r16		;PortD0,D1 als Eingang
;
	ldi	r16,rxbuff		;Pointer fr Empfangspuffer
	mov	rxpoi1,r16		;Pointer 1 setzen
	mov	rxpoi2,r16		;Pointer 2 setzen
	ldi	r16,rstext		;Pointer fr Textpuffer
	mov	rspoin,r16		;Pointer setzen
	clr	precnt			;Interrupt-Vorteiler lschen
	clr	buzcn1			;Buzzer-Timeout-Zhler 1 lsch.
	clr	buzcn2			;Buzzer-Timeout-Zhler 2 lsch.
	clr	flags			;Flags lschen
	clr	dimtio			;Anzeige-Timeout-Zhler lschen
	clr	lastal			;letzte Alarmnummer lschen
	clr	dimode			;Anzeigemodus auf Temperatur
	clr	xh			;High-Byte fr Interrupt=0
	clr	yh			;High-Byte fr Pointer=0 setzen
;
	ldi	yl,tembuf		;Zeiger auf Temperaturwerte
	ldi	r17,8*5			;8 Werte mit je 5 Zeichen
	ldi	r16,' '			;mit Leerzeichen fllen
sta020:	st	y+,r16			;Speicherzelle lschen
	dec	r17			;alle Zellen gelscht?
	brne	sta020			;nein -> Schleife
;
	ldi	r16,60			;Anfangswert fr Meldungs-
	mov	msgtio,r16		;Timeout auf 60s setzen
;
	ldi	r16,(clock/(16*baud))-1	;Baudrate RS-232 auf 9600 Baud
	out	ubrr,r16		;Baudratenwert setzen
	ldi	r16,(1<<rxcie)+(1<<rxen)
	out	ucr,r16			;RX mit Interrupt aktivieren
;
	ldi	r16,(1<<ctc1)+(1<<cs11)	;Timer-Reset bei Interrupt
	out	tccr1b,r16		;und Vorteiler=8 setzen
	ldi	r16,low(timval)		;Timer1-Compare-Wert setzen
	ldi	r17,high(timval)
	out	ocr1ah,r17		;ausgeben
	out	ocr1al,r16
	ldi	r16,(1<<ocie1a)		;Timer1-Compare-Interrupt
	out	timsk,r16		;aktivieren
;
; LCD initialisieren und Sonderzeichen programmieren
;
;	rcall	w50ms			;Warteschleife 50ms
;	clr	r17			;LCD in Befehlsmodus setzen
;	ldi	r16,0x30		;Initialisierung
;	rcall	lcdout			;ausgeben					;;;<--> Die komplette Init-Prozedur knnte theoretisch fast weggelassen werden, 
;	rcall	w04ms			;Warteschleife 4,1ms				;;;<--> der KS0073 initialisiert sich von selbst beinahe so wie wir ihn bentigen.
;	rcall	lcdout			;Init nochmals ausgeben				;;;<--> Trtozdem wird die init prozedur angepasst!
;	rcall	w100my			;Warteschleife 100s
;	rcall	lcdw40			;Init nochmals ausgeben
;	ldi	r16,0x38		;8-Bit-IF, 2-Line, 5x7 Font
;	rcall	lcdw40			;ausgeben
;	ldi	r16,0x0c		;Display ein, kein Cursor
;	rcall	lcdw40			;ausgeben
;	ldi	r16,0x06		;Increment, kein Scolling
;	rcall	lcdw40			;ausgeben


;;;<--> Angepasste Init Prozedur fr LCD mit KS0073!!!!!!!!!!!

	rcall	w50ms			;50ms warten
	clr	r17			;Befehlsmodus == RS -> Low !
	ldi	r16,0b00111100		;RS und R/W sind low, 8 bit mode, 2 line mode, RE HIGH!
	rcall	lcdw40			;ausgeben und im anschluss 40 ys warten!
					;Hier muss Init (warum auch immer, danke KS0073) nicht nochmals aushgegeben werden!
	ldi	r16,0b00001001		;extended function set ausgeben, 5 dot font width; 4 LINE MODE
	rcall	lcdw40			;ausgeben und mind. 39ys warten!
	ldi	r16,0b00111000		;function set wieder 8 bit, 2 line, RE wieder LOW!
	rcall	lcdw40			;ausgeben und wieder 40ys warten.
	ldi	r16,0b00001100		;Display on, cursor off, blink off
	rcall	lcdw40			;ausgeben und im Anschluss 40 ys warten.
	ldi	r16,0x01		;Clear display!
	rcall	lcdout			;ausgeben
	rcall	w02ms			;Mind. 1,51ms warten, Funktion wartet ca. 1,64ms
	ldi	r16,0b00000110		;Increment mode, auto shift (=scrolling?) off
	rcall	lcdw40			;ausgeben, Warten laut Datenblatt nicht notwendig, zur Sicherheit dennoch
	
;;;<-->Ende der angepassten Init Prozedur fr LCD mit KS0073!!!!!!
;;;<-->4 Line Mode verwendet, da scheinbar der KS0066(u) im 2 Line Mode auch 4 Line Anweisungen versteht. Da sich aber der KS0073 explizit in einen 4Line Modus setzen lsst, tun wir dies.




;
	ldi	r16,0x40		;LCD-Adresse der Sonderzeichen			
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse setzen
	ldi	r19,24			;24 Bytes fr 3 Sonderzeichen			
	ldi	r17,1			;LCD in Datenmodus setzen
	ldi	zl,low(2*chars)		;Zeiger auf Zeichen-Bitmaps
	ldi	zh,high(2*chars)	;(Grad-Zeichen und Pfeile)
sta100:	lpm				;Datenbyte aus Flash holen
	mov	r16,r0			;und kopieren
	adiw	zl,1			;Zeiger auf nchstes Datenbyte
	rcall	lcdw40			;an LCD ausgeben
	dec	r19			;alle Zeilen ausgegeben?
	brne	sta100			;nein -> Schleife
;
	ldi	r16,0x01		;Befehl: Anzeige lschen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms				;;;<--> Eigentlicht wrden 1,53 ms reichen -> Dem Display aber auch Zeit nach den Daten geben?
;
; Begrungstext und Versionsnummer ausgeben
;
	

	clr	r19			;Zeichenzhler=0 setzen				
	ldi	zl,low(2*sttext)	;Zeiger auf Text setzen
	ldi	zh,high(2*sttext)	
sta200:	cpi	r19,20			;Beginn Zeile 2?
	breq	sta220			;ja -> LCD-Adresse setzen
	cpi	r19,40			;Beginn Zeile 3?
	breq	sta230			;ja -> LCD-Adresse setzen
	cpi	r19,60			;Beginn Zeile 4?
	breq	sta240			;ja -> LCD-Adresse setzen
	cpi	r19,80			;Textende erreicht?
	breq	sta260			;ja -> Ende
sta210:	lpm				;Zeichen aus Programmspeicher
	mov	r16,r0			;holen und bereitlegen
	adiw	zl,1			;Zeiger auf nchstes Zeichen
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	inc	r19			;Zeichenzhler erhhen
	rjmp	sta200			;Schleife
;
sta220:	ldi	r16,0xa0		;LCD-Adresse fr Zeile 2			;;;<-->Hier wird vom Afang 40H (zweite Zeile 2line mode) auf 20H (4line mode) umgestellt!
	rjmp	sta250									;;;<-->Also statt 0xc0 zu 0xa0
sta230:	ldi	r16,0xc0		;LCD-Adresse fr Zeile 3			;;;<--> Hier wird 0x14 zu 0x40
	rjmp	sta250
sta240:	ldi	r16,0xe0		;LCD-Adresse fr Zeile 4			;;;<--> Hier wird 0x54 zu 0x60
sta250:	clr	r17			;LCD in Befehlsmodus setzen			;;;<--> Der KS0073 wird im 2 Line Modus mit IE HIGH Zeilenmssig anders angesteuert (Siehe Datenblatt) als 												;;;<-->der  
											;;;<--> KS0066(u). Der Einfachheit halber daher Umstellung auf echten 4 Line Mode.
	rcall	lcdw40			;Zeichen ausgeben
	rjmp	sta210			;Schleife
;
sta260:	ldi	r19,200			;Warteschleife 200*50ms (10s)			
sta270:	rcall	w50ms
	dec	r19
	wdr				;Watchdog zurcksetzen
	brne	sta270
;
	clr	r17			;LCD in Befehlsmodus setzen
	ldi	r16,0x01		;Befehl: Anzeige lschen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms


;
; Anzeige-Tabelle aus EEPROM lesen und Werte prfen
;
	ldi	r16,etempm		;EEPROM-Adresse der Tabelle			
	ldi	r18,8			;8 Werte auslesen
	ldi	yl,temmap		;Zeiger auf Anzeige-Tabelle
sta300:	rcall	eeread			;EEPROM lesen
	cpi	r17,17			;Sensornummer>16?
	brcc	sta900			;ja -> Fehler
	st	y+,r17			;sonst Sensornummer speichern			
	inc	r16			;EEPROM-Adresse +1
	dec	r18			;alle Zeichen gelesen?
	brne	sta300			;nein -> Schleife
;
; Alarm-Invertierungs-Tabelle aus EEPROM lesen und Werte prfen
;
	ldi	r16,ealmin		;EEPROM-Adresse der Tabelle
	ldi	r18,4			;4 Werte auslesen
	ldi	yl,alminv		;Zeiger auf Werte-Tabelle
sta400:	rcall	eeread			;EEPROM lesen
	cpi	r17,2			;Wert>1?
	brcc	sta900			;ja -> EEPROM-Fehler
	st	y+,r17			;sonst Zeichen speichern
	inc	r16			;EEPROM-Adresse +1
	dec	r18			;alle Zeichen gelesen?
	brne	sta400			;nein -> Schleife
;
; Temperatur- und Alarmtexte auf Plausibilitt prfen
;
	ldi	r16,etempt		;EEPROM-Adresse der Sensortexte
	ldi	r18,(8*4)+(4*12)	;8 Temp. und 4 Alarme prfen
sta700:	rcall	eeread			;Zeichen aus EEPROM holen
	cpi	r17,' '			;Zeichen < Leerzeichen?
	brcs	sta900			;ja -> EEPROM-Fehler
	cpi	r17,255			;Zeichencode=255?
	breq	sta900			;ja -> EEPROM-Fehler
	inc	r16			;nchste EEPROM-Adresse
	dec	r18			;alle Texte geprft?
	brne	sta700			;nein -> Schleife
;
	sei				;Interrupts aktivieren
	rjmp	main
;
; Fehlertext bei unplausiblen EEPROM-Daten ausgeben, LED bleibt an
;
sta900:	ldi	r16,0xa3		;LCD-Adresse fr Zeile 2, Pos 3					;;;<--> 0xc3 Zeile 2 Pos 3 (wir beginnen bei 0 zu zhlen. anders als das datenblatt) bei 														;;;<-->unserem Display: 0x23 -> mit vorangestellter
													;;;<--> binrer 1 um den SET DDRAM ADDRESS befehl zu komplettieren: 0xa3
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Ausgabe
	ldi	r19,14			;14 Zeichen ausgeben
	ldi	zl,low(2*eftext)	;Zeiger auf Text setzen
	ldi	zh,high(2*eftext)
sta910:	lpm				;Zeichen aus Programmspeicher
	mov	r16,r0			;holen und bereitlegen
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	adiw	zl,1			;Zeiger auf nchstes Zeichen
	dec	r19			;alle Zeichen ausgegeben?
	brne	sta910			;nein -> Schleife
;
sta920:	wdr				;Watchdog zurcksetzen
	rjmp	sta920			;Endlos-Schleife
;
; ---------------------
; Hauptprogrammschleife
; ---------------------
;
; Alarm-LED bei Fehler oder Alarm blinken lassen
; 40ms aus / 460ms ein (2Hz) bei Alarm
; 40ms ein / 960ms aus (1Hz) bei Fehler oder Temperatur-Alarm
;
main:	ldi	yl,almbuf		;Zeiger auf Alarmwerte
	clr	r16			;Zhler fr Alarme
ma0010:	ld	r17,y+			;Alarmwert holen
	tst	r17			;Alarm aktiv?
	breq	ma0020			;nein -> weiter
	inc	r16			;sonst Alarmzhler erhhen
ma0020:	cpi	yl,almbuf+4		;alle Werte verarbeitet?
	brne	ma0010			;nein -> Schleife
	tst	r16			;sind Alarme aktiv?
	brne	ma0040			;ja -> LED zeigt Alarm
	bst	flags,4			;Fehler-Flag gesetzt?					
	brts	ma0070			;ja -> LED zeigt Fehler					
	rcall	limtst			;Limit-berschreitungen?
	brne	ma0070			;ja -> LED zeigt Fehler
	sbi	portd,led		;sonst LED ausschalten
	rjmp	ma0100
;

;;;<--> Part fr ALARM LED SIGNAL !!!!!!


ma0040:	mov	r16,precnt		;Vorteiler holen
	cpi	r16,25			;Wert>=25?
	brcs	ma0050			;nein -> weiter						
	subi	r16,25			;sonst Wert-25
ma0050:	cpi	r16,2			;Wert>1? (entspricht 40ms)
	brcc	ma0060			;ja -> LED ein
	sbi	portd,led		;sonst LED ausschalten
	rjmp	ma0100
ma0060:	cbi	portd,led		;LED einschalten
	rjmp	ma0100


;;;<--> Part fr FEHLER LED SIGNAL!!


ma0070:	mov	r16,precnt		;Vorteiler holen
	cpi	r16,2			;Wert>1? (entspricht 40ms)
	brcc	ma0080			;ja -> LED aus
	cbi	portd,led		;sonst LED einschalten
	rjmp	ma0100
ma0080:	sbi	portd,led		;LED ausschalten
;
; Tastenabfrage und Anzeige entsprechend Modus
;
ma0100:	mov	r16,dimode		;Anzeigemodus holen
	bst	flags,2			;Taste gedrckt?
	brtc	ma0120			;nein -> weiter
	bst	flags,3			;wurde Taste schon quittiert?
	brts	ma0120			;ja -> weiter
	sbr	flags,0x29		;Taste quittieren/nderungsflag
	inc	r16			;Anzeigemodus erhhen
	cpi	r16,2			;Wert>1?
	brcs	ma0110			;nein -> berspringen
	clr	r16			;sonst wieder auf 0 setzen
ma0110:	mov	dimode,r16		;Anzeigemodus wieder speichern
	ser	r17			;Anzeigetimeout inaktiv setzen
	mov	dimtio,r17		;Timeout-Zhler speichern
ma0120:	rcall	limtst			;Limit-berschreitungen?
	breq	ma0150			;nein -> berspringen
	cp	secnt2,secnt1		;neue Anzeige (Blinken)?
	breq	ma0150			;nein -> berspringen
	mov	secnt2,secnt1		;sonst Sekundenzhler speichern
	sbr	flags,0x01		;nderungs-Flag setzen
ma0150:	bst	flags,0			;nderungs-Flag gesetzt?
	brtc	ma0160			;nein -> keine neue Anzeige
	cbr	flags,0x01		;sonst nderungs-Flag lschen
	mov	r16,dimode		;Anzeigemodus holen
	tst	r16			;Anzeigemodus Temperatur?
	brne	ma0160			;nein -> weiter testen
	rcall	temper			;sonst Temperaturdaten ausgeben
	rjmp	ma0200
ma0160:	cpi	r16,1			;Alarmanzeige?
	brne	ma0200			;nein -> berspringen
	bst	flags,5			;nderungsflag 2 gesetzt?
	brtc	ma0200			;nein -> keine neue Anzeige
	cbr	flags,0x20		;sonst nderungsflag 2 lschen
	rcall	alarm			;Alarme ausgeben
;
; Timeouts fr Temperaturen bearbeiten
;
ma0200:	ldi	yl,tempto		;Zeiger auf Timeout-Zhler
	clr	r18			;aktuelle Temp-Sensornummer
ma0210:	ld	r19,y			;Timeout-Zhler holen
	tst	r19			;Zhler=0?
	brne	ma0230			;nein -> berspringen
	ldi	r17,5			;5 Leerzeichen ausgeben
	ldi	zl,tembuf		;Zeiger auf Temperaturwerte
	clr	zh
	mov	r16,r18			;Sensornummer holen
	add	r16,r16			;verdoppeln
	add	r16,r16			;nochmals verdoppeln (x4)
	add	r16,r18			;insgesamt x5 als Offset
	add	zl,r16			;fr Temp-Tabelle addieren
	ldi	r16,' '			;Leerzeichen laden
ma0220:	st	z+,r16			;und in Werte-Tabelle speichern
	dec	r17			;alle Zeichen ausgegeben?
	brne	ma0220			;nein -> Schleife
	dec	r19			;Timeout-Zhler auf 255 setzen
	sbr	flags,1			;nderungs-Flag setzen
	st	y,r19			;Timeout-Zhler wieder speich.
ma0230:	inc	yl			;Zeiger auf nchsten Tab.-Platz
	inc	r18			;nchsten Sensor bearbeiten
	cpi	r18,8			;alle Sensoren bearbeitet?
	brne	ma0210			;nein -> Schleife
;
; Timeout fr Meldungen bearbeiten
;
	tst	msgtio			;Meldungs-Timeout-Zhler=0?
	brne	ma0240			;nein -> berspringen
	dec	msgtio			;Meldungs-Timeout inaktiv setz.
	sbr	flags,0b10001		;Fehler-/nderungs-Flag setzen
	clr	dimode			;Anzeige auf Temperatur setzen
;
; Timeout fr Anzeige-Rckstellung auf Temperatur bearbeiten					
;
ma0240:	ldi	yl,almbuf		;Zeiger auf Alarmwerte
	clr	r16			;Zhler fr Alarme
ma0250:	ld	r17,y+			;Alarmwert holen
	tst	r17			;Alarm aktiv?
	breq	ma0260			;nein -> weiter
	inc	r16			;sonst Alarmzhler erhhen
ma0260:	cpi	yl,almbuf+4		;alle Werte verarbeitet?
	brne	ma0250			;nein -> Schleife
	bst	flags,4			;Fehler-Flag gesetzt?
	brtc	ma0270			;nein -> weiter
	inc	r16			;sonst Alarmzhler erhhen
ma0270:	tst	r16			;sind Alarme aktiv?
	brne	ma0280			;ja -> weiter
;
	mov	r16,dimode		;sonst Anzeigemodus holen
	tst	r16			;Temperaturanzeige?
	breq	ma0280			;ja -> berspringen
	mov	r16,dimtio		;Anzeige-Timeout-Zhler holen
	cpi	r16,0xff		;Zhler aktiv?
	brne	ma0280			;ja -> weiter
	ldi	r16,timdim		;sonst Timeout-Wert holen
	mov	dimtio,r16		;und Zhler neu setzen
ma0280:	tst	dimtio			;Anzeige-Timeout-Zhler=0?
	brne	ma0300			;nein -> berspringen
	dec	dimtio			;Anzeige-Timeout inaktiv setzen
	clr	dimode			;Anzeige auf Temperatur setzen
	sbr	flags,1			;nderungs-Flag setzen
;
; Buzzer entsprechend der Alarmnummer steuern:
; 4x 80ms Tonfolge pro Sekunde (Alarm 1)
; 3x 80ms Tonfolge pro Sekunde (Alarm 2)
; 2x 80ms Tonfolge pro Sekunde (Alarm 3)
; 1x 80ms Tonfolge pro Sekunde (Alarm 4)
; 2x 20ms Tonfolge pro Sekunde (Temperatur-Alarm)
;
ma0300:	mov	r16,buzcn1		;Buzzer-Timeout-Zhler 1 holen
	cpi	r16,0xff		;Timeout-Zhler aktiv?
	breq	ma0400			;nein -> berspringen
	cpi	r16,0			;Zhler=0?
	brne	ma0310			;nein -> bearbeiten, sonst
	dec	buzcn1			;Timeout-Zhler auf 255 setzen
	cbi	portd,buzzer		;Buzzer abschalten
	rjmp	ma0500
;
ma0310:	mov	r19,lastal		;letzte Alarmnummer holen
	ldi	r18,32			;Tonfolgenlnge setzen (640ms)
ma0320:	tst	r19			;enspricht Alarm=Tonfolge?
	breq	ma0330			;ja -> Tonausgabe
	subi	r18,8			;sonst Tonfolgenlnge -160ms
	dec	r19			;nchste Alarmnummer
	rjmp	ma0320			;Schleife
;
ma0330:	ldi	r17,timbuz		;Buzzer-Timeout-Wert holen
	sub	r17,r16			;aktuellen Zhlerwert subtrah.
ma0340:	cpi	r17,50			;Ergebnis<50 (<1 Sekunde)?
	brcs	ma0350			;ja -> Ton ausgeben
	subi	r17,50			;sonst Wert um 50 vermindern
	rjmp	ma0340			;Schleife
;
ma0350:	cp	r17,r18			;Wert>Tonfolgenlnge?
	brcc	ma0360			;ja -> Buzzer ausschalten
	bst	r17,2			;Wert 4-7, 12-15, 20-23 usw.?
	brts	ma0360			;ja -> Buzzer ausschalten
	sbi	portd,buzzer		;sonst Buzzer einschalten
	rjmp	ma0500
ma0360:	cbi	portd,buzzer		;Buzzer ausschalten
	rjmp	ma0500
;
ma0400:	mov	r16,buzcn2		;Buzzer-Timeout-Zhler 2 holen
	cpi	r16,0xff		;Timeout-Zhler aktiv?
	breq	ma0500			;nein -> berspringen
	cpi	r16,0			;Zhler=0?
	brne	ma0410			;nein -> bearbeiten, sonst
	dec	buzcn2			;Timeout-Zhler auf 255 setzen
	cbi	portd,buzzer		;Buzzer abschalten
	rjmp	ma0500
;
ma0410:	ldi	r17,timbuz		;Buzzer-Timeout-Wert holen
	sub	r17,r16			;aktuellen Zhlerwert subtrah.
ma0420:	cpi	r17,25			;Ergebnis<25 (<0,5 Sekunden)?
	brcs	ma0430			;ja -> Ton ausgeben
	subi	r17,25			;sonst Wert um 25 vermindern
	rjmp	ma0420			;Schleife
;
ma0430:	cpi	r17,0			;Wert=0 (Beginn einer 1/2 Sek)?
	brne	ma0440			;nein -> Buzzer ausschalten
	sbi	portd,buzzer		;sonst Buzzer einschalten
	rjmp	ma0500
ma0440:	cbi	portd,buzzer		;Buzzer ausschalten
;
; Empfangspuffer lesen
;
ma0500:	rcall	getchr			;Zeichen aus Empfangspuffer
	tst	r17			;holen, neues Zeichen?
	brne	ma0510			;nein -> weiter suchen
	cpi	r16,lf			;LF (10) empfangen?
	breq	ma0510			;ja -> ignorieren
	cpi	r16,cr			;CR (13) empfangen?
	breq	ma1000			;ja -> Puffer auswerten
;
	mov	yl,rspoin		;Zeiger auf Textpuffer
	cpi	yl,rstext+8		;Puffer voll?
	breq	ma0510			;ja -> Zeichen verwerfen
	st	y+,r16			;Zeichen im Puffer ablegen
	mov	rspoin,yl		;Zeiger wieder speichern
ma0510:	rjmp	main			;Schleife
;
ma1000:	mov	r16,rspoin		;Textpuffer-Zeiger holen
	cpi	r16,rstext+7		;Temperatur-Daten empfangen?
	breq	ma2000			;ja -> bearbeiten
	cpi	r16,rstext+3		;Alarm-Daten empfangen?
	brne	ma1010			;nein -> weiter testen
	rjmp	ma3000			;sonst Alarm bearbeiten
ma1010:	cpi	r16,rstext+8		;Sensorbelegung empfangen?	
	brne	ma1100			;nein -> Ende
	rjmp	ma4000			;sonst Sensorbelegung bearbeit.
;
ma1100:	ldi	r16,rstext		;sonst Pointer fr Textpuffer
	mov	rspoin,r16		;wieder auf Anfang setzen
	rjmp	main			;Schleife
;
; empfangene Zeile (Temperaturwert) auf Plausibilitt prfen
;
ma2000:	ldi	yl,rstext		;Zeiger auf Textpuffer
	ld	r16,y+			;erstes Zeichen holen
	cpi	r16,'1'			;Zeichen<"1"?
	brcs	ma1100			;ja -> Fehler
	cpi	r16,'h'+1		;Zeichen>"h"?
	brcc	ma1100			;ja -> Fehler
	cpi	r16,'9'			;Zeichen<"9"?
	brcs	ma2010			;ja -> weiter (Sensor 1-8)
	cpi	r16,'a'			;Zeichen<"a"?
	brcs	ma1100			;ja -> Fehler
ma2010:	ld	r16,y+			;zweites Zeichen holen
	cpi	r16,':'			;Doppelpunkt?
	brne	ma1100			;nein -> Fehler
	ld	r16,y+			;drittes Zeichen holen
	cpi	r16,' '			;Leerzeichen?
	breq	ma2020			;ja -> weiter
	cpi	r16,'-'			;Minus-Zeichen?
	breq	ma2020			;ja -> weiter
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	ma1100			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen>"9"?
	brcc	ma1100			;ja -> Fehler
ma2020:	ld	r16,y+			;viertes Zeichen holen
	cpi	r16,' '			;Leerzeichen?
	breq	ma2030			;ja -> weiter
	cpi	r16,'-'			;Minus-Zeichen?
	breq	ma2030			;ja -> weiter
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	ma1100			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen>"9"?
	brcc	ma1100			;ja -> Fehler
ma2030:	ld	r16,y+			;fnftes Zeichen holen
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	ma1100			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen>"9"?
	brcc	ma1100			;ja -> Fehler
	ld	r16,y+			;sechstes Zeichen holen
	cpi	r16,'.'			;Punkt?
	breq	ma2040			;ja -> weiter
	cpi	r16,','			;Komma?
	brne	ma1100			;nein -> Fehler
ma2040:	ld	r16,y+			;siebentes Zeichen holen
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	ma1100			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen>"9"?
	brcc	ma1100			;ja -> Fehler
;
; prfen, ob Sensornummer in Anzeige-Tabelle vorhanden ist
;
	ldi	yl,rstext		;Zeiger auf Textpuffer
	ld	r16,y+			;erstes Zeichen holen
	cpi	r16,'9'			;Sensorgruppe 1-8?
	brcc	ma2050			;nein -> weiter
	subi	r16,'1'			;in Sensornummer 0-7 umwandeln
	rjmp	ma2060
ma2050:	subi	r16,'a'-8		;in Sensornummer 8-15 umwandeln
ma2060:	ldi	zl,temmap
	clr	zh			;Zeiger auf Anzeige-Tabelle
	clr	r18			;Tabellenplatz-Zhler
ma2070:	ld	r17,z			;Tabelle lesen
	dec	r17			;Wert korrigieren
	cp	r16,r17			;Sensornummer gefunden?
	breq	ma2080			;ja -> weiter
	adiw	zl,1			;Zeiger auf nchst. Tabellenpl.
	inc	r18			;nchster Tabellenplatz
	cpi	r18,8			;alle Tabellenpltze gelesen?
	brne	ma2070			;nein -> Schleife
	rjmp	ma1100			;sonst Datenpaket verwerfen

; Temperatur-Daten sind ok, in SRAM-Tabelle speichern

ma2080:	mov	r20,r18			;Tabellenpl. zwischenspeichern
	adiw	yl,1			;Zeiger auf Text setzen,
	ldi	r17,5			;5 Textzeich. kopieren/ausgeben
	ldi	zl,tembuf		;Zeiger auf Temperaturwerte
	clr	zh
	add	r18,r18			;Sensornummer verdoppeln
	add	r18,r18			;nochmals verdoppeln (x4)
	add	r18,r20			;insgesamt x5 als Offset fr
	add	zl,r18			;Temp-Tabelle addieren
	mov	r21,zl			;Zeiger auf Temp-Wert sichern
ma2100:	ld	r16,y+			;Zeichen aus Textpuffer holen
	ld	r18,z			;alten Wert aus Tabelle holen
	cp	r16,r18			;hat sich der Wert gendert?
	breq	ma2110			;nein -> weiter
	sbr	flags,1			;sonst nderungs-Flag setzen
ma2110:	st	z+,r16			;neuen Wert speichern
	dec	r17			;alle Zeichen ausgegeben?
	brne	ma2100			;nein -> Schleife
;
; Temperatur auf Limit-ber- oder Unterschreitung prfen
;
	bst	flags,0			;nderungs-Flag gesetzt?
	brtc	ma2400			;nein -> weiter
	ldi	yl,temlim		;Zeiger auf Limit-Flags
	add	yl,r20			;Sensornummer addieren
	mov	zl,r21			;Zeiger auf Temp-Wert holen
	rcall	asc_hx			;Temperatur-Wert in Hex wandeln
	ldi	r19,100			;100 addieren fr vereinfachte
	add	r18,r19			;Rechung mit negativen Werten
;
	ldi	r16,eliml		;EEPROM-Adresse Minimalwerte
	add	r16,r20			;Sensornummer addieren
	rcall	eeread			;EERPOM lesen
	cpi	r17,128			;Alarmberwachung aktiv?
	breq	ma2120			;nein -> Prfung berspringen
	add	r17,r19			;100 addieren
	cp	r18,r17			;Minimalwert unterschritten?
	brcs	ma2140			;ja -> auswerten
;
ma2120:	ldi	r16,elimh		;EEPROM-Adresse Maximalwerte
	add	r16,r20			;Sensornummer addieren
	rcall	eeread			;EERPOM lesen
	cpi	r17,128			;Alarmberwachung aktiv?
	breq	ma2130			;nein -> Prfung berspringen
	add	r17,r19			;100 addieren
	cp	r18,r17			;Maximalwert berschritten?
	brcc	ma2150			;ja -> auswerten
;	
ma2130:	clr	r16			;sonst Limit-Flag lschen
	rjmp	ma2300
;
ma2140:	ld	r16,y			;aktuelle Limit-Flags holen
	bst	r16,0			;unt. Limit-Flag schon gesetzt?
	brts	ma2400			;ja -> berspringen
	ldi	r17,timbuz		;Timeout-Wert fr Buzzer
	mov	buzcn2,r17		;setzen
	ldi	r16,1			;Limit-Flag fr Unterschreitung
	rjmp	ma2300			;setzen
ma2150:	ld	r16,y			;aktuelle Limit-Flags holen
	bst	r16,1			;ob. Limit-Flag schon gesetzt?
	brts	ma2400			;ja -> berspringen
	ldi	r17,timbuz		;Timeout-Wert fr Buzzer
	mov	buzcn2,r17		;setzen
	ldi	r16,2			;Limit-Flag fr berschreitung
;
ma2300:	st	y,r16			;und wieder speichern
ma2400:	ldi	yl,tempto		;Zeiger auf Timeout-Zhler
	add	yl,r20			;Sensornummer addieren (Offset)
	ldi	r16,timtmp		;Timeout-Zhler neu setzen
	st	y,r16			;und speichern
	ldi	r16,timmsg		;Meldungs-Timeout-Zhler neu
	mov	msgtio,r16		;setzen
	cbr	flags,0b10000		;Fehler-Flag lschen
	sbr	flags,0b00001		;nderungs-Flag setzen
	rjmp	ma1100			;Text-Zeiger zurcksetzen, Ende
;
; empfangene Zeile (Alarmmeldung) auf Plausibilitt prfen
;
ma3000:	ldi	yl,rstext		;Zeiger auf Textpuffer
	ld	r16,y+			;erstes Zeichen holen
	cpi	r16,'A'			;Zeichen<"A"?
	brcs	ma3100			;ja -> Fehler
	cpi	r16,'D'+1		;Zeichen>"D"?
	brcc	ma3100			;ja -> Fehler
	ld	r16,y+			;zweites Zeichen holen
	cpi	r16,':'			;Doppelpunkt?
	brne	ma3100			;nein -> Fehler
	ld	r16,y+			;drittes Zeichen holen
	cpi	r16,'0'			;Zeichen="0"?
	breq	ma3010			;ja -> bearbeiten
	cpi	r16,'1'			;Zeichen="1"?
	brne	ma3100			;nein -> Fehler
;
; Alarm-Daten sind ok, in SRAM-Tabelle speichern
;
ma3010:	ldi	yl,rstext		;Zeiger auf Textpuffer
	ld	r16,y+			;erstes Zeichen holen
	andi	r16,0x0f		;in Ziffernwert wandeln
	dec	r16			;korrigieren (=Alarmummer)
	adiw	yl,1			;Zeiger auf Text setzen,
	ldi	zl,almbuf		;Zeiger auf Alarm-Werte
	clr	zh
	add	zl,r16			;Tabellenplatz berechnen
	ld	r17,y			;Zeichen aus Textpuffer holen
	andi	r17,0x0f		;in Ziffernwert wandeln
	ldi	yl,alminv		;Zeiger auf Alarm-Invertierung
	add	yl,r16			;Tabellenplatz berechnen
	ld	r18,y			;Invertierungswert holen
	eor	r17,r18			;bei Bedarf Alarm invertieren
	ld	r18,z			;alten Wert aus Tabelle holen
	cp	r17,r18			;hat sich der Wert gendert?
	breq	ma3020			;nein -> weiter
	sbr	flags,0x20		;sonst nderungsflag 2 setzen
	cpi	r17,1			;wurde Alarm ausgelst?
	brne	ma3020			;nein -> weiter
	ldi	r18,1			;sonst Anzeigemodus Alarm
	mov	dimode,r18		;setzen
	ldi	r18,timbuz		;Timeout-Wert fr Buzzer
	mov	buzcn1,r18		;setzen
	mov	lastal,r16		;letzte Alarmnummer speichern
	ser	r18			;Anzeigetimeout inaktiv setzen
	mov	dimtio,r18		;Timeout-Zhler wieder speich.
ma3020:	st	z,r17			;neuen Wert speichern
	ldi	r16,timmsg		;Meldungs-Timeout-Zhler neu
	mov	msgtio,r16		;setzen
	cbr	flags,0b10000		;Fehler-Flag lschen
	sbr	flags,0b00001		;nderungs-Flag setzen
ma3100:	rjmp	ma1100			;Text-Zeiger zurcksetzen, Ende
;
; empfangene Zeile (Sensorbelegung) auf Plausibilitt prfen
;
ma4000:	ldi	yl,rstext		;Zeiger auf Textpuffer
ma4010:	ld	r16,y+			;Zeichen aus Puffer holen
	cpi	r16,'0'			;Zeichen<"0"?
	brcs	ma3100			;ja -> Fehler
	cpi	r16,'G'+1		;Zeichen>"G"?
	brcc	ma3100			;ja -> Fehler
	cpi	r16,'9'+1		;Zeichen<="9"?
	brcs	ma4020			;ja -> weiter
	cpi	r16,'A'			;Zeichen<"A"?
	brcs	ma3100			;ja -> Fehler
ma4020:	cpi	yl,rstext+8		;Pufferende erreicht?
	brne	ma4010			;nein -> Schleife
;
; Sensorbelegung ist ok, Daten sofort auf LCD ausgeben
;
	ldi	r16,0x01		;Befehl: Anzeige lschen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms
	ldi	r16,0xa3		;LCD-Adresse fr Zeile 2, Pos 3					;;;<--> 0xc3 Zeile 2 Pos 3 bei unserem Display: 0x23 -> mit vorangestellter
													;;;<--> binrer 1 um den SET DDRAM ADDRESS befehl zu komplettieren: 0xa3
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	r19,14			;14 Zeichen ausgeben
	ldi	zl,low(2*senmap)	;Zeiger auf Text setzen
	ldi	zh,high(2*senmap)
ma4030:	lpm				;Zeichen aus Programmspeicher
	mov	r16,r0			;holen und bereitlegen
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	adiw	zl,1			;Zeiger auf nchstes Zeichen
	dec	r19			;alle Zeichen ausgegeben?
	brne	ma4030			;nein -> Schleife
;
	ldi	r16,0xc6		;LCD-Adresse fr Zeile 3, Pos 6					;;;<--> 0x9a zeile 3 Pos 6 bei KS0073: 0x46, mit vorangest. bin 1: 0xc6
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	yl,rstext		;Zeiger auf Textpuffer
ma4040:	ld	r16,y+			;Zeichen aus Puffer holen
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	cpi	yl,rstext+8		;Pufferende erreicht?
	brne	ma4040			;nein -> Schleife
;
	ldi	r16,2			;Anzeigemodus auf Sensorbele-
	mov	dimode,r16		;gung setzen
	ldi	r16,timsem		;Timeout-Wert fr Sensorbeleg.
	mov	dimtio,r16		;laden und Zhler neu setzen
	ldi	r16,timmsg		;Meldungs-Timeout-Zhler neu
	mov	msgtio,r16		;setzen
	cbr	flags,0b10000		;Fehler-Flag lschen
	sbr	flags,0b00001		;nderungs-Flag setzen
	rjmp	ma1100			;Text-Zeiger zurcksetzen, Ende
;
; Unterprogramm zur Ausgabe der Temperaturdaten auf LCD
;
temper:	bst	flags,4			;Fehler-Flag gesetzt?
	brtc	tem010			;nein -> weiter
	rjmp	tem200			;sonst Fehlertext anzeigen
tem010:	clr	r20			;Positions-Zhler=0
tem100:	mov	r16,r20			;Positions-Zhler kopieren
	clr	r17			;Dummy fr Adress-Berechnung
	ldi	zl,low(2*lcdtab)	;Zeiger auf LCD-Adress-Tabelle
	ldi	zh,high(2*lcdtab)
	add	zl,r16			;Tabellenplatz berechnen
	adc	zh,r17			;bertrag addieren
	lpm				;Tabelleneintrag holen und
	mov	r16,r0			;kopieren
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	r16,etempt		;EEPROM-Adresse Sensortexte
	mov	r17,r20			;Sensorzhler holen
	add	r17,r17			;verdoppeln
	add	r17,r17			;nochmals verdoppeln
	mov	r21,r17			;und sichern
	add	r21,r20			;Offset fr Tempwert (*5)
	add	r16,r17			;Adresse berechnen
	ldi	r19,4			;4 Zeichen lesen
tem110:	rcall	eeread			;EEPROM lesen
	push	r16			;EEPROM-Adresse sichern
	mov	r16,r17			;EEPROM-Daten kopieren
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	pop	r16			;EEPROM-Adresse zurckholen
	inc	r16			;nchste Adresse
	dec	r19			;alle Zeichen ausgegeben?
	brne	tem110			;nein -> Schleife

	ldi	yl,temlim		;Zeiger auf Limit-Tabelle
	add	yl,r20			;Tabellenplatz berechnen
	ld	r15,y			;Limit-Flag holen
	tst	r15			;Limit-berschreitung?
	breq	tem130			;nein -> normale Temp-Ausgabe
	mov	r16,secnt1		;Sekundenzhler holen
	andi	r16,0x01		;Sekunde geradzahlig?
	breq	tem130			;ja -> normale Temp-Ausgabe
	ldi	r19,6			;sonst 6 Leerzeichen ausgeben
	ldi	r16,' '			;Leerzeichencode
tem120:	rcall	lcdw40			;Zeichen ausgeben
	dec	r19			;alle Zeichen ausgegeben?
	brne	tem120			;nein -> Schleife
	rjmp	tem160			;nchsten Sensor bearbeiten
;
tem130:	ldi	r19,5			;5 Zeichen ausgeben (Tempwert)
	ldi	yl,tembuf		;Zeiger auf Temperaturwerte
	add	yl,r21			;Tabellenoffset addieren
tem140:	ld	r16,y+			;Zeichen holen, Adresse +1
	rcall	lcdw40			;Zeichen ausgeben
	dec	r19			;alle Zeichen ausgegeben?
	brne	tem140			;nein -> Schleife
	cpi	r16,' '			;letztes Zeichen= Leerzeichen?
	breq	tem150			;ja -> kein Grad-Zeichen
	clr	r16			;Sonderzeichen 0 (Grad)
	tst	r15			;Limit-Unterschreitung?
	breq	tem150			;ja -> berspringen
	mov	r16,r15			;sonst Limit-Flag holen
	andi	r16,3			;Flag=Sonderzeichen-Code
tem150:	rcall	lcdw40			;ausgeben
tem160:	inc	r20			;Zhler auf nchsten Sensor
	cpi	r20,8			;alle Tabellenpltze ausgegeb?
	brne	tem100			;nein -> Schleife
	cbr	flags,1			;nderungs-Flag lschen
	ret
;
tem200:	ldi	r16,0x01		;Befehl: Anzeige lschen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms
	ldi	r16,0xa1		;LCD-Adresse fr Zeile 2, Pos 1					;;;<--> 0xc1 zeile 2 Pos 1 bei uns: 0x21, mit bin null fr ddram adressbef: 0xa1
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	r19,18			;18 Zeichen ausgeben
	ldi	zl,low(2*msgerr)	;Zeiger auf Text setzen
	ldi	zh,high(2*msgerr)
tem210:	lpm				;Zeichen aus Programmspeicher
	mov	r16,r0			;holen und bereitlegen
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	adiw	zl,1			;Zeiger auf nchstes Zeichen
	dec	r19			;alle Zeichen ausgegeben?
	brne	tem210			;nein -> Schleife
	ret
;
; Unterprogramm zur Alarmanzeige auf LCD
;
alarm:	ldi	r16,0x01		;Befehl: Anzeige lschen
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdout			;ausgeben
	rcall	w15ms			;Warteschleife 15,2ms
;
	ldi	yl,almbuf		;Zeiger auf Alarmmeldungen
	ldi	r19,4			;4 Alarme prfen
	clr	r16			;Anzahl der Alarme feststellen
alm100:	ld	r17,y+			;Alarmwert holen
	tst	r17			;Alarmmeldung aktiv?
	breq	alm110			;nein -> weiter
	inc	r16			;Alarmzhler erhhen
alm110:	dec	r19			;alle Werte geprft?
	brne	alm100			;nein -> Schleife
;
	tst	r16			;sind Alarme aktiv?
	breq	alm300			;nein -> anzeigen
	ldi	r22,1			;Meldungen ab Zeile 2 schreiben
	cpi	r16,3			;sind mehr als 2 Alarme aktiv?
	brcs	alm120			;nein -> weiter
	dec	r22			;sonst in Zeile 1 beginnen
alm120:	ldi	yl,almbuf		;Zeiger auf Alarmmeldungen
	clr	r21			;Alarmnummernzhler, Anfang=0
;
alm130:	ld	r20,y+			;Alarmwert holen
	tst	r20			;Alarm aktiv?
	breq	alm170			;nein -> keine Ausgabe
	mov	r16,r22			;Zeilennummer holen
	rcall	lcdadr			;LCD-Adresse ermitteln
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	zl,low(2*almakt)	;Zeiger auf Alarm-Text
	ldi	zh,high(2*almakt)
	ldi	r19,7			;7 Zeichen ausgeben
	ldi	r17,1			;LCD in Datenmodus setzen
alm150:	lpm				;Textzeichen holen
	mov	r16,r0			;und kopieren
	rcall	lcdw40			;Zeichen ausgeben
	adiw	zl,1			;Zeiger erhhen
	dec	r19			;alle Zeichen ausgegeben?
	brne	alm150			;nein -> Schleife
;
	ldi	r20,ealmtx		;Zeiger auf EEPROM-Alarmtexte
	mov	r16,r21			;Alarmnummer holen
	add	r16,r16			;x2
	add	r16,r21			;x3
	add	r16,r16			;x6
	add	r16,r16			;x12
	add	r20,r16			;Offset addieren
	ldi	r19,12			;12 Zeichen ausgeben
alm160:	mov	r16,r20			;EEPROM-Adresse setzen
	rcall	eeread			;EEPROM lesen
	mov	r16,r17			;Zeichen kopieren
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	inc	r20			;nchste EEPROM-Adresse
	dec	r19			;alle Zeichen ausgegeben?
	brne	alm160			;nein -> Schleife
	inc	r22			;auf nchste Zeile setzen
;
alm170:	inc	r21			;Alarmnummer erhhen
	cpi	r21,4			;alle Alarme bearbeitet?
	brne	alm130			;nein -> Schleife
	ret
;
alm300:	ldi	r16,0xa4		;LCD-Adresse fr Zeile 2, Pos 4						;;;<-->0xc4 2 Pos 4 bei uns: 0x24 -> 0xa4
	clr	r17			;LCD in Befehlsmodus setzen
	rcall	lcdw40			;Adresse ausgeben
	ldi	r19,12			;12 Zeichen ausgeben
	ldi	zl,low(2*noalrm)	;Zeiger auf Text setzen
	ldi	zh,high(2*noalrm)
alm310:	lpm				;Zeichen aus Programmspeicher
	mov	r16,r0			;holen und bereitlegen
	ldi	r17,1			;LCD in Datenmodus setzen
	rcall	lcdw40			;Zeichen ausgeben
	adiw	zl,1			;Zeiger auf nchstes Zeichen
	dec	r19			;alle Zeichen ausgegeben?
	brne	alm310			;nein -> Schleife
	ret
;
; Unterprogramm zum Ermitteln der LCD-Adresse
; Register: r0,r17,z
;	    r16 Zeilennummer -> LCD-Adresse
;
lcdadr:	clr	r17			;Dummy fr Adress-Berechnung
	ldi	zl,low(2*lcdtab)	;Zeiger auf LCD-Adress-Tabelle
	ldi	zh,high(2*lcdtab)
	add	zl,r16			;Tabellenplatz berechnen
	adc	zh,r17			;bertrag addieren
	lpm				;Tabelleneintrag holen und
	mov	r16,r0			;kopieren
	ret
;
; Unterprogramm zum Auslesen eines Zeichens aus dem RS-232-
; Empfangspuffer
; Register: y
;	    r16 gelesenes Zeichen
;	    r17 0= Zeichen gltig, 0xff= kein Zeichen im Puffer
;
getchr:	cp	rxpoi1,rxpoi2		;sind Zeichen im Puffer?
	brne	get010			;ja -> weiter
	ser	r17			;kein Zeichen im Puffer
	ret
get010:	cli				;Int vorbergehend sperren
	mov	yl,rxpoi2		;Empfangspuffer-Pointer2 laden
	ld	r16,y+			;Byte laden, Pointer+1
	andi	yl,0x9f			;bertrag korrigieren
	mov	rxpoi2,yl		;Pointer2 wieder speichern
	sei				;Int wieder freigeben
	clr	r17			;Zeichen ist gltig
	ret
;
; Unterprogramm LCD-Ausgabe eines Zeichens
; Register: r16 LCD-Daten/Befehlsbyte
;	    r17 0=Befehlsbyte, 1=Datenbyte
;
lcdout:	out	portb,r16		;Daten/Befehlsbyte ausgeben
	tst	r17			;Datenbyte?
	brne	lcd010			;ja -> weiter
	cbi	portd,2			;LCD auf Befehlsmodus setzen
	rjmp	lcd020
lcd010:	sbi	portd,2			;LCD auf Datenmodus setzen
lcd020:	nop
	sbi	portd,3			;E auf High setzen
	nop
	cbi	portd,3			;E wieder auf Low setzen
	ret				;(Datenbernahme)
;
; Unterprogramm LCD-Ausgabe eines Zeichens und Warteschleife 40s
; Register: r16 LCD-Daten/Befehlsbyte
;	    r17 0=Befehlsbyte, 1=Datenbyte
;
lcdw40:	rcall	lcdout			;Zeichen ausgeben und
	rjmp	w040my			;Warteschleife 40s
;
; Unterprogramm Warteschleife ca. 50ms
; Register: r18 Zhler High
;
w50ms:	ldi	r18,0			;High-Zhler setzen
	clr	delayc			;Low-Zhler setzen
w50ms1:	dec	delayc			;Low-Zhler-1
	brne	w50ms1			;Schleife
	dec	r18			;High-Zhler-1
	brne	w50ms1			;Schleife
	ret
;
; Unterprogramm Warteschleife ca. 15,2ms
; Register: r18 Zhler High
;
w15ms:	ldi	r18,243
	mov	delayc,r18		;Low-Zhler setzen
	ldi	r18,79			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 4,1ms
; Register: r18 Zhler High
;
w04ms:	ldi	r18,73
	mov	delayc,r18		;Low-Zhler setzen
	ldi	r18,22			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 1.64ms
; Register: r18 Zhler High
;
w02ms:	ldi	r18,130
	mov	delayc,r18		;Low-Zhler setzen
	ldi	r18,9			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 100s
; Register: r18 Zhler High
;
w100my:	ldi	r18,129
	mov	delayc,r18		;Low-Zhler setzen
	ldi	r18,1			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm Warteschleife ca. 40s
; Register: r18 Zhler High
;
w040my:	ldi	r18,49
	mov	delayc,r18		;Low-Zhler setzen
	ldi	r18,1			;High-Zhler setzen
	rjmp	w50ms1
;
; Unterprogramm zum Lesen eines Byte aus dem EEPROM
; Register: r16 EEPROM-Adresse
;	    r17 EEPROM-Daten
;
eeread:	sbic	eecr,eewe		;luft Schreibzyklus?
	rjmp	eeread			;ja - warten
	out	eearl,r16		;EEPROM-Adresse setzen
	sbi	eecr,eere		;EEPROM lesen aktivieren
	in	r17,eedr		;Daten lesen
	ret
;
; Unterprogramm zur Konvertierung eines Temperaturwertes von ASCII in
; einen Hexadezimalwert
; Register: r16,r17
;	    z   Zeiger auf Temperaturwert (ASCII)
;	    r18	Temperaturwert in Hex
;
asc_hx:	clr	r18			;Startwert=0 setzen
	ld	r17,z			;Vorzeichen/Hunderter holen
	cpi	r17,'1'			;Ziffer 1?
	brne	asc010			;nein -> weiter
	ldi	r18,100			;sonst Startwert=100 setzen
asc010:	ldd	r17,z+1			;Zehner holen
	cpi	r17,' '			;Leerzeichen?
	breq	asc030			;ja -> weiter zum Einer
	cpi	r17,'-'			;Minus-Zeichen?
	breq	asc030			;ja -> weiter zum Einer
	subi	r17,0x30		;ASCII in Zahl umwandeln
	ldi	r16,10			;wird fr Zehner bentigt
asc020:	tst	r17			;Zahl=0?
	breq	asc030			;ja -> weiter zum Einer
	add	r18,r16			;sonst 10 addieren
	dec	r17			;Zhler vermindern
	rjmp	asc020			;Schleife
asc030:	ldd	r17,z+2			;Einer holen
	subi	r17,0x30		;ASCII in Zahl umwandeln
	add	r18,r17			;Einer addieren
	ld	r17,z			;nochmals Vorzeichen/Hunderter
	cpi	r17,'-'			;Minus-Zeichen?
	brne	asc060			;nein -> weiter
asc040:	neg	r18			;sonst Zweierkomplement bilden
	ldd	r17,z+4			;Zehntel holen
	cpi	r17,'0'			;Zehntel=0?
	breq	asc050			;ja -> Ergebnis ok
	dec	r18			;sonst Ergebnis korrigieren
asc050:	ret				;Ende
asc060:	ldd	r17,z+1			;nochmals Zehner holen
	cpi	r17,'-'			;Minus-Zeichen?
	breq	asc040			;ja -> Zweierkomplement
	ret				;sonst Ende
;
; Unterprogramm zum Test der Limit-Flags auf Temperatur-ber- oder
; Unterschreitung
; Register r16,r17,y
;	   Z-Flag Ergebnis: 0=ok, 1=Limit-berschreitung
;
limtst:	ldi	yl,temlim		;Zeiger auf Limit-Flags
	clr	r16			;Summenzhler lschen
lim010:	ld	r17,y+			;Flag holen
	or	r16,r17			;und mit Summe verknpfen
	cpi	yl,temlim+8		;alle Werte verarbeitet?
	brne	lim010			;nein -> Schleife
	tst	r16			;Limit-berschreitungen?
	ret
;
; Interrupt-Routine "UART Empfang komplett"
; empfangenes Byte im Ringpuffer speichern
; Register: x
;
rxcint:	in	insreg,sreg		;SREG sichern
	in	itemp,udr		;empfangenes Byte lesen
	mov	xl,rxpoi1		;Empfangspuffer-Pointer1 laden
	st	x+,itemp		;Byte speichern, Pointer+1
	andi	xl,0x9f			;bertrag korrigieren
	mov	rxpoi1,xl		;Pointer1 wieder speichern
	out	sreg,insreg		;SREG wiederherstellen
	reti
;
; Interrupt-Routine bei Timer1-Compare, wird etwa alle 20 ms ausgefhrt
; Register: x
;
; Taste lesen und Flags setzen
;
timer1:	in	insreg,sreg		;SREG sichern
	in	itemp,pind		;Taste lesen
	bst	itemp,1			;Taste gedrckt?
	brts	tim020			;nein -> weiter
	bst	flags,1			;war Taste bereits gedrckt?
	brtc	tim010			;nein -> Tastenflag 1 setzen
	sbr	flags,0x04		;sonst Tastenflag 2 setzen
	rjmp	tim100
tim010:	sbr	flags,0x02		;Tastenflag 1 setzen
	rjmp	tim100
;
tim020:	bst	flags,1			;war Taste vorher gedrckt?
	brtc	tim030			;nein -> Tastenflag 2 lschen
	cbr	flags,0x02		;sonst Tastenflag 1 lschen
	rjmp	tim100
tim030:	cbr	flags,0x0c		;Tastenflags 2 und 3 lschen
;
; Timeout 1 und 2 fr Buzzer steuern
;
tim100:	mov	itemp,buzcn1		;Timeout-Zhler holen
	tst	itemp			;Zhlerwert=0?
	breq	tim110			;ja -> berspringen
	cpi	itemp,0xff		;Zhlerwert=255?
	breq	tim110			;ja -> berspringen
	dec	itemp			;sonst Zhlerwert-1
	mov	buzcn1,itemp		;Zhlerwert wieder speichern
;
tim110:	mov	itemp,buzcn2		;Timeout-Zhler holen
	tst	itemp			;Zhlerwert=0?
	breq	tim120			;ja -> berspringen
	cpi	itemp,0xff		;Zhlerwert=255?
	breq	tim120			;ja -> berspringen
	dec	itemp			;sonst Zhlerwert-1
	mov	buzcn2,itemp		;Zhlerwert wieder speichern
;
; Vorteiler bearbeiten
;
tim120:	inc	precnt			;Vorteiler erhhen
	mov	itemp,precnt		;Vorteiler kopieren
	cpi	itemp,50		;Endwert erreicht?
	brne	timend			;nein -> Ende
	clr	precnt			;sonst Vorteiler zurcksetzen
	inc	secnt1			;und Sekundenzhler erhhen
;
; Timeouts fr Temperaturen, Meldungen und Anzeige steuern
;
	ldi	xl,tempto		;Zeiger auf Temp-Timeout-Werte
tim200:	ld	itemp,x			;Zhlerwert holen
	tst	itemp			;Zhlerwert=0?
	breq	tim210			;ja -> berspringen
	cpi	itemp,0xff		;Zhlerwert=255?
	breq	tim210			;ja -> berspringen
	dec	itemp			;sonst Zhlerwert-1
tim210:	st	x+,itemp		;Zhlerwert wieder speichern
	cpi	xl,tempto+8		;alle Werte bearbeitet?
	brne	tim200			;nein -> Schleife
;
	mov	itemp,msgtio		;Meldungs-Timeout-Zhler holen
	tst	itemp			;Zhlerwert=0?
	breq	tim230			;ja -> berspringen
	cpi	itemp,0xff		;Zhlerwert=255?
	breq	tim230			;ja -> berspringen
	dec	itemp			;sonst Zhlerwert-1
	mov	msgtio,itemp		;Zhler wieder speichern
;
tim230:	mov	itemp,dimtio		;Anzeige-Timeout-Zhler holen
	tst	itemp			;Zhlerwert=0?
	breq	tim240			;ja -> berspringen
	cpi	itemp,0xff		;Zhlerwert=255?
	breq	tim240			;ja -> berspringen
	dec	itemp			;sonst Zhlerwert-1
	mov	dimtio,itemp		;Zhlerwert wieder speichern
tim240:	wdr				;Watchdog zurcksetzen
;
timend:	out	sreg,insreg		;SREG wiederherstellen
	reti
;
; Verschiedene Texte und Tabellen
;
sttext:	.db	"Anzeigemodul 1 fuer "	;Begrungstext, 4 Zeilen
	.db	"Temperaturmesssystem"
	.db	"for KS0073 LCD CTRLR"
	.db	"(c) Scott-Falk Huehn"


;
eftext:	.db	"EEPROM-Fehler "	;Anzeige bei EEPROM-Fehler
;
senmap:	.db	"Sensorbelegung"	;Anzeige Sensorbelegung
;
msgerr:	.db	"kein Datenempfang "	;Anzeige bei Daten-Timeout
;
noalrm:	.db	"keine Alarme"		;Anzeige wenn alarmfrei
;
almakt:	.db	"Alarm:  "		;Anzeige bei Alarm
;
lcdtab:	.db	0x80,0xa0,0xc0,0xe0	;LCD-Adresse fr Werte 1-4			;;;<-->Auch hier mssen die Adressen wieder angepasst werden!
	.db	0x8a,0xaa,0xca,0xea	;LCD-Adresse fr Werte 5-8			;;;<-->0x80 ist okay, 0xc0 wird zu: 0xa0, 0x94 wird zu: 0xc0; und d4 wird zu e0 <--> 0x8a bleibt, ist Position 												;;;<-->11 Zeile 1; 0xca wird zu 0xaa; 94 wird zu ca; de wird zu ea.
;
chars:	.db	0x10,0,0,0,0,0,0,0	;Bitmap Sonderz.0 (Grad)
	.db	0x10,0,4,4,0x15,0x0e,4,0;Bitmap Sonderz.1 (Pfeil 1)
	.db	0x10,0,4,0x0e,0x15,4,4,0;Bitmap Sonderz.2 (Pfeil 2)
;
;
.include "konfiguration.inc"
;.include "beispiel-01-08.inc"
;.include "beispiel-09-16.inc"
;
